//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2025 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.typechecker.postchk;

import static org.eclipse.escet.common.java.Lists.list;
import static org.eclipse.escet.common.java.Lists.listc;
import static org.eclipse.escet.common.java.Maps.mapc;

import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.function.Predicate;

import org.eclipse.escet.cif.common.CifCollectUtils;
import org.eclipse.escet.cif.common.CifTextUtils;
import org.eclipse.escet.cif.common.CifValueUtils;
import org.eclipse.escet.cif.metamodel.cif.Specification;
import org.eclipse.escet.cif.metamodel.cif.automata.Assignment;
import org.eclipse.escet.cif.metamodel.cif.automata.Automaton;
import org.eclipse.escet.cif.metamodel.cif.automata.Edge;
import org.eclipse.escet.cif.metamodel.cif.automata.ElifUpdate;
import org.eclipse.escet.cif.metamodel.cif.automata.IfUpdate;
import org.eclipse.escet.cif.metamodel.cif.automata.Location;
import org.eclipse.escet.cif.metamodel.cif.automata.Update;
import org.eclipse.escet.cif.metamodel.cif.declarations.DiscVariable;
import org.eclipse.escet.cif.metamodel.cif.expressions.ContVariableExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.DiscVariableExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.Expression;
import org.eclipse.escet.cif.metamodel.cif.expressions.InputVariableExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.ProjectionExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.TupleExpression;
import org.eclipse.escet.cif.metamodel.cif.functions.InternalFunction;
import org.eclipse.escet.cif.typechecker.ErrMsg;

/**
 * Additional type checking for {@link DiscVariable discrete variables}, during the 'post' type checking phase. It
 * checks the following:
 * <ul>
 * <li>Whether discrete variables are effectively constant.</li>
 * </ul>
 */
public class DiscVariablePostChecker {
    /** The post check environment to use. */
    private final CifPostCheckEnv env;

    /**
     * Constructor for the {@link DiscVariablePostChecker} class.
     *
     * @param env The post check environment to use.
     */
    public DiscVariablePostChecker(CifPostCheckEnv env) {
        this.env = env;
    }

    /**
     * Check a specification.
     *
     * @param spec The specification to check.
     */
    public void check(Specification spec) {
        // Discrete variables can only be defined in automata. Check each automaton.
        for (Automaton aut: CifCollectUtils.collectAutomata(spec, list())) {
            check(aut);
        }
    }

    /**
     * Check an automaton.
     *
     * @param aut The automaton to check.
     */
    private void check(Automaton aut) {
        // Get discrete variables.
        List<DiscVariable> discVars = CifCollectUtils.collectDiscVariables(aut, list());

        // If there are no variables, there is nothing to check. Prevent doing more work than is needed.
        if (discVars.isEmpty()) {
            return;
        }

        // Check for each variable whether it could be a constant. First, determine the variables with a single initial
        // value. If there are none, there is no work to do.
        Predicate<DiscVariable> hasSingleInitValue = v -> v.getValue() == null || v.getValue().getValues().size() == 1;
        List<DiscVariable> varsToConsider = discVars.stream().filter(hasSingleInitValue).toList();
        if (varsToConsider.isEmpty()) {
            return;
        }

        // Get for each variable that is to be considered its relevant information, such as its initial value.
        Map<DiscVariable, VarInfo> varInfos = mapc(varsToConsider.size());
        List<InternalFunction> typeFuncs = listc(0);
        for (DiscVariable discVar: varsToConsider) {
            Expression initialValue = (discVar.getValue() == null)
                    ? CifValueUtils.getDefaultValue(discVar.getType(), typeFuncs)
                    : discVar.getValue().getValues().get(0);
            varInfos.put(discVar, new VarInfo(initialValue));
        }

        // Walk over the edges, to see whether the variables are assigned and in what way. We try to see if the initial
        // value of the variable is always preserved. We look for cases where it is assigned itself or its initial
        // value. We do not consider all cases, and therefore may miss some effectively-constant discrete variables. Any
        // variable detected as being effectively-constant, is really effectively constant. We may thus miss some cases,
        // but we don't have any false positives.
        for (Location loc: aut.getLocations()) {
            for (Edge edge: loc.getEdges()) {
                for (Update update: edge.getUpdates()) {
                    processUpdate(update, varInfos);
                }
            }
        }

        // Report discrete variables that we know for sure are effectively constant.
        for (Entry<DiscVariable, VarInfo> entry: varInfos.entrySet()) {
            // If a variable is assigned a value that is not itself or its initial value, or a value for which we
            // couldn't determine what exactly is assigned, we never report a warning. This prevents false positives.
            // Note that we also don't update the self and initial value assignments information anymore once such an
            // 'other' value has been assigned, and as such that information can be incomplete. We thus only use that
            // information if no 'other' value is assigned.
            DiscVariable discVar = entry.getKey();
            VarInfo varInfo = entry.getValue();
            if (!varInfo.isAssignedOtherValue) {
                String reasonTxt;
                if (varInfo.isAssignedItself && varInfo.isAssignedInitialValue) {
                    reasonTxt = "it is only ever assigned itself or its initial value";
                } else if (varInfo.isAssignedItself) {
                    reasonTxt = "it is only ever assigned itself";
                } else if (varInfo.isAssignedInitialValue) {
                    reasonTxt = "it is only ever assigned its initial value";
                } else {
                    reasonTxt = "it is never assigned";
                }
                env.addProblem(ErrMsg.DISC_VAR_EFFECTIVELY_CONSTANT, discVar.getPosition(),
                        CifTextUtils.getAbsName(discVar), reasonTxt);
            }
        }
    }

    /**
     * Update the variable information based on the given update.
     *
     * @param update The update to process.
     * @param varInfos The variable information for variables that are to be considered. Is updated in-place.
     */
    private void processUpdate(Update update, Map<DiscVariable, VarInfo> varInfos) {
        if (update instanceof Assignment assignment) {
            // Process the assignment by recursively considering its addressable and corresponding value.
            processAssignment(assignment.getAddressable(), assignment.getValue(), varInfos);
        } else if (update instanceof IfUpdate ifUpdate) {
            // Recursively consider the 'then', 'elif' and 'else' updates.
            for (Update thenUpdate: ifUpdate.getThens()) {
                processUpdate(thenUpdate, varInfos);
            }

            for (ElifUpdate elifUpdate: ifUpdate.getElifs()) {
                for (Update thenUpdate: elifUpdate.getThens()) {
                    processUpdate(thenUpdate, varInfos);
                }
            }

            for (Update elseUpdate: ifUpdate.getElses()) {
                processUpdate(elseUpdate, varInfos);
            }
        } else {
            throw new AssertionError("Unknown update: " + update);
        }
    }

    /**
     * Update the variable information based on the given (partial) assignment.
     *
     * @param addressable The addressable of the (partial) assignment.
     * @param value The value of the (partial) assignment. May be {@code null} if the exact value is not known.
     * @param varInfos The variable information for variables that are to be considered. Is updated in-place.
     */
    private void processAssignment(Expression addressable, Expression value, Map<DiscVariable, VarInfo> varInfos) {
        // Unfold projections.
        while (addressable instanceof ProjectionExpression projAddr) {
            // Process the addressable being projected. We don't know the exact value in this case.
            addressable = projAddr.getChild();
            value = null;
        }

        // Handle other addressables.
        if (addressable instanceof ContVariableExpression) {
            // Skip continuous variables being assigned, as it is not relevant for this check.
            return;
        } else if (addressable instanceof InputVariableExpression) {
            // Skip input variables being assigned, as it is reported elsewhere in the type checker as being invalid.
            return;
        } else if (addressable instanceof TupleExpression tupleAddr) {
            // Process the addressables that are part of the tuple. We don't know the exact value in this case.
            for (Expression fieldAddr: tupleAddr.getFields()) {
                processAssignment(fieldAddr, null, varInfos);
            }
        } else if (addressable instanceof DiscVariableExpression discVarAddr) {
            // Update the variable's information based on this assignment, but:
            // - Only process assignments to variables being considered.
            // - Only process assignments to variables that don't yet have assignments to other values. This is a
            //   performance optimization. We don't report a warning in such cases, so no need to spend more effort on
            //   processing assignments to it.
            DiscVariable discVar = discVarAddr.getVariable();
            VarInfo varInfo = varInfos.get(discVar);
            if (varInfo != null && !varInfo.isAssignedOtherValue) {
                if (value == null) {
                    // Don't know the exact value, so can't be sure the initial value is preserved.
                    varInfo.isAssignedOtherValue = true;
                } else if (value instanceof DiscVariableExpression valueRef && valueRef.getVariable() == discVar) {
                    // Variable is assigned to itself. This preserves its initial value.
                    varInfo.isAssignedItself = true;
                } else if (CifValueUtils.areStructurallySameExpression(varInfo.initialValue, value)) {
                    // Variable is assigned its initial value. This preserves its initial value.
                    varInfo.isAssignedInitialValue = true;
                } else {
                    // Variable is assigned something else, so we can't be sure the initial value is preserved.
                    varInfo.isAssignedOtherValue = true;
                }
            }
        } else {
            throw new AssertionError("Unexpected addressable: " + addressable);
        }
    }

    /** Information about a discrete variable. */
    private static class VarInfo {
        /** The single initial value of the variable. */
        final Expression initialValue;

        /** Whether the variable has self-assignments. */
        boolean isAssignedItself = false;

        /** Whether the variable is assigned its initial value. */
        boolean isAssignedInitialValue = false;

        /**
         * Whether the variable is assigned any other value than its initial value, or a value for which it could not be
         * determined exactly what value is assigned.
         */
        boolean isAssignedOtherValue = false;

        /**
         * Constructor for the {@link VarInfo} class.
         *
         * @param initialValue The single initial value of the variable.
         */
        VarInfo(Expression initialValue) {
            this.initialValue = initialValue;
        }
    }
}
