/*
 * Decompiled with CFR 0.152.
 */
package io.moquette.broker;

import io.moquette.broker.ISslContextCreator;
import io.moquette.broker.config.IConfig;
import io.netty.handler.ssl.ClientAuth;
import io.netty.handler.ssl.SslContext;
import io.netty.handler.ssl.SslContextBuilder;
import io.netty.handler.ssl.SslProvider;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.security.GeneralSecurityException;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.util.Collections;
import java.util.Objects;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.TrustManagerFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class DefaultMoquetteSslContextCreator
implements ISslContextCreator {
    private static final Logger LOG = LoggerFactory.getLogger(DefaultMoquetteSslContextCreator.class);
    private final IConfig props;

    DefaultMoquetteSslContextCreator(IConfig props) {
        this.props = Objects.requireNonNull(props);
    }

    @Override
    public SslContext initSSLContext() {
        LOG.info("Checking SSL configuration properties...");
        String keyPassword = this.props.getProperty("key_manager_password");
        if (keyPassword == null || keyPassword.isEmpty()) {
            LOG.warn("The key manager password is null or empty. The SSL context won't be initialized.");
            return null;
        }
        try {
            SslContextBuilder contextBuilder;
            SslProvider sslProvider = this.getSSLProvider();
            KeyStore ks = this.loadKeyStore();
            switch (sslProvider) {
                case JDK: {
                    contextBuilder = DefaultMoquetteSslContextCreator.builderWithJdkProvider(ks, keyPassword);
                    break;
                }
                case OPENSSL: 
                case OPENSSL_REFCNT: {
                    contextBuilder = DefaultMoquetteSslContextCreator.builderWithOpenSSLProvider(ks, keyPassword);
                    break;
                }
                default: {
                    LOG.error("unsupported SSL provider {}", (Object)sslProvider);
                    return null;
                }
            }
            String sNeedsClientAuth = this.props.getProperty("need_client_auth", "false");
            if (Boolean.valueOf(sNeedsClientAuth).booleanValue()) {
                DefaultMoquetteSslContextCreator.addClientAuthentication(ks, contextBuilder);
            }
            contextBuilder.sslProvider(sslProvider);
            SslContext sslContext = contextBuilder.build();
            LOG.info("The SSL context has been initialized successfully.");
            return sslContext;
        }
        catch (IOException | GeneralSecurityException ex) {
            LOG.error("Unable to initialize SSL context.", ex);
            return null;
        }
    }

    private KeyStore loadKeyStore() throws IOException, GeneralSecurityException {
        String jksPath = this.props.getProperty("jks_path");
        LOG.info("Initializing SSL context. KeystorePath = {}.", (Object)jksPath);
        if (jksPath == null || jksPath.isEmpty()) {
            LOG.warn("The keystore path is null or empty. The SSL context won't be initialized.");
            return null;
        }
        String keyStorePassword = this.props.getProperty("key_store_password");
        if (keyStorePassword == null || keyStorePassword.isEmpty()) {
            LOG.warn("The keystore password is null or empty. The SSL context won't be initialized.");
            return null;
        }
        String ksType = this.props.getProperty("key_store_type", "jks");
        KeyStore keyStore = KeyStore.getInstance(ksType);
        LOG.info("Loading keystore. KeystorePath = {}.", (Object)jksPath);
        try (InputStream jksInputStream = this.jksDatastore(jksPath);){
            keyStore.load(jksInputStream, keyStorePassword.toCharArray());
        }
        return keyStore;
    }

    private static SslContextBuilder builderWithJdkProvider(KeyStore ks, String keyPassword) throws GeneralSecurityException {
        LOG.info("Initializing key manager...");
        KeyManagerFactory kmf = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
        kmf.init(ks, keyPassword.toCharArray());
        LOG.info("Initializing SSL context...");
        return SslContextBuilder.forServer(kmf);
    }

    private static SslContextBuilder builderWithOpenSSLProvider(KeyStore ks, String keyPassword) throws GeneralSecurityException {
        for (String alias : Collections.list(ks.aliases())) {
            if (!ks.entryInstanceOf(alias, KeyStore.PrivateKeyEntry.class)) continue;
            PrivateKey key = (PrivateKey)ks.getKey(alias, keyPassword.toCharArray());
            Certificate[] chain = ks.getCertificateChain(alias);
            X509Certificate[] certChain = new X509Certificate[chain.length];
            System.arraycopy(chain, 0, certChain, 0, chain.length);
            return SslContextBuilder.forServer(key, certChain);
        }
        throw new KeyManagementException("the SSL key-store does not contain a private key");
    }

    private static void addClientAuthentication(KeyStore ks, SslContextBuilder contextBuilder) throws NoSuchAlgorithmException, KeyStoreException {
        LOG.warn("Client authentication is enabled. The keystore will be used as a truststore.");
        TrustManagerFactory tmf = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
        tmf.init(ks);
        contextBuilder.clientAuth(ClientAuth.REQUIRE);
        contextBuilder.trustManager(tmf);
    }

    private SslProvider getSSLProvider() {
        String providerName = this.props.getProperty("ssl_provider", SslProvider.JDK.name());
        try {
            return SslProvider.valueOf(providerName);
        }
        catch (IllegalArgumentException e) {
            LOG.warn("unknown SSL Provider {}, falling back on JDK provider", (Object)providerName);
            return SslProvider.JDK;
        }
    }

    private InputStream jksDatastore(String jksPath) throws FileNotFoundException {
        URL jksUrl = this.getClass().getClassLoader().getResource(jksPath);
        if (jksUrl != null) {
            LOG.info("Starting with jks at {}, jks normal {}", (Object)jksUrl.toExternalForm(), (Object)jksUrl);
            return this.getClass().getClassLoader().getResourceAsStream(jksPath);
        }
        LOG.warn("No keystore has been found in the bundled resources. Scanning filesystem...");
        File jksFile = new File(jksPath);
        if (jksFile.exists()) {
            LOG.info("Loading external keystore. Url = {}.", (Object)jksFile.getAbsolutePath());
            return new FileInputStream(jksFile);
        }
        throw new FileNotFoundException("The keystore file does not exist. Url = " + jksFile.getAbsolutePath());
    }
}

