/*******************************************************************************
 * Copyright (c) 2000, 2022 IBM Corporation and others.
 *
 * This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.jdt.internal.compiler.parser;

import java.util.HashMap;
import java.util.Map;

/**
 * IMPORTANT NOTE: These constants are dedicated to the internal Scanner implementation.
 * It is mirrored in org.eclipse.jdt.core.compiler public package where it is API.
 * The mirror implementation is using the backward compatible ITerminalSymbols constant
 * definitions (stable with 2.0), whereas the internal implementation uses TerminalTokens
 * which constant values reflect the latest parser generation state.
 */
/**
 * Maps each terminal symbol in the java-grammar into a unique integer.
 * This integer is used to represent the terminal when computing a parsing action.
 *
 * Disclaimer : These constant values are generated automatically using a Java
 * grammar, therefore their actual values are subject to change if new keywords
 * were added to the language (for instance, 'assert' is a keyword in 1.4).
 */
public enum TerminalToken {

	// special tokens not part of grammar - not autogenerated
							TokenNameInvalid(-1),
							TokenNameNotAToken(0),
							TokenNameWHITESPACE(1000),
							TokenNameCOMMENT_LINE(1001),
							TokenNameCOMMENT_BLOCK(1002),
							TokenNameCOMMENT_JAVADOC(1003),
							TokenNameSingleQuoteStringLiteral(1004),
							TokenNameCOMMENT_MARKDOWN(1005),

	// BEGIN_AUTOGENERATED_REGION
							TokenNameIdentifier(27),
							TokenNameabstract(51),
							TokenNameassert(90),
							TokenNameboolean(103),
							TokenNamebreak(91),
							TokenNamebyte(104),
							TokenNamecase(105),
							TokenNamecatch(106),
							TokenNamechar(107),
							TokenNameclass(69),
							TokenNamecontinue(92),
							TokenNameconst(135),
							TokenNamedefault(86),
							TokenNamedo(93),
							TokenNamedouble(108),
							TokenNameelse(117),
							TokenNameenum(83),
							TokenNameextends(102),
							TokenNamefalse(40),
							TokenNamefinal(52),
							TokenNamefinally(116),
							TokenNamefloat(109),
							TokenNamefor(94),
							TokenNamegoto(136),
							TokenNameif(95),
							TokenNameimplements(131),
							TokenNameimport(110),
							TokenNameinstanceof(14),
							TokenNameint(111),
							TokenNameinterface(82),
							TokenNamelong(112),
							TokenNamenative(53),
							TokenNamenew(36),
							TokenNamenon_sealed(54),
							TokenNamenull(41),
							TokenNamepackage(101),
							TokenNameprivate(55),
							TokenNameprotected(56),
							TokenNamepublic(57),
							TokenNamereturn(96),
							TokenNameshort(113),
							TokenNamestatic(38),
							TokenNamestrictfp(58),
							TokenNamesuper(34),
							TokenNameswitch(62),
							TokenNamesynchronized(39),
							TokenNamethis(35),
							TokenNamethrow(87),
							TokenNamethrows(118),
							TokenNametransient(59),
							TokenNametrue(42),
							TokenNametry(97),
							TokenNamevoid(114),
							TokenNamevolatile(60),
							TokenNamewhile(88),
							TokenNamemodule(115),
							TokenNameopen(119),
							TokenNamerequires(120),
							TokenNametransitive(126),
							TokenNameexports(121),
							TokenNameopens(122),
							TokenNameto(132),
							TokenNameuses(123),
							TokenNameprovides(124),
							TokenNamewith(133),
							TokenNameIntegerLiteral(43),
							TokenNameLongLiteral(44),
							TokenNameFloatingPointLiteral(45),
							TokenNameDoubleLiteral(46),
							TokenNameCharacterLiteral(47),
							TokenNameStringLiteral(48),
							TokenNameTextBlock(49),
							TokenNamePLUS_PLUS(8),
							TokenNameMINUS_MINUS(9),
							TokenNameEQUAL_EQUAL(19),
							TokenNameLESS_EQUAL(15),
							TokenNameGREATER_EQUAL(16),
							TokenNameNOT_EQUAL(20),
							TokenNameLEFT_SHIFT(18),
							TokenNameRIGHT_SHIFT(12),
							TokenNameUNSIGNED_RIGHT_SHIFT(13),
							TokenNamePLUS_EQUAL(70),
							TokenNameMINUS_EQUAL(71),
							TokenNameMULTIPLY_EQUAL(72),
							TokenNameDIVIDE_EQUAL(73),
							TokenNameAND_EQUAL(74),
							TokenNameOR_EQUAL(75),
							TokenNameXOR_EQUAL(76),
							TokenNameREMAINDER_EQUAL(77),
							TokenNameLEFT_SHIFT_EQUAL(78),
							TokenNameRIGHT_SHIFT_EQUAL(79),
							TokenNameUNSIGNED_RIGHT_SHIFT_EQUAL(80),
							TokenNameOR_OR(26),
							TokenNameAND_AND(25),
							TokenNamePLUS(1),
							TokenNameMINUS(2),
							TokenNameNOT(63),
							TokenNameREMAINDER(5),
							TokenNameXOR(22),
							TokenNameAND(21),
							TokenNameMULTIPLY(4),
							TokenNameOR(23),
							TokenNameTWIDDLE(64),
							TokenNameDIVIDE(6),
							TokenNameGREATER(17),
							TokenNameLESS(7),
							TokenNameLPAREN(28),
							TokenNameRPAREN(29),
							TokenNameLBRACE(61),
							TokenNameRBRACE(33),
							TokenNameLBRACKET(11),
							TokenNameRBRACKET(100),
							TokenNameSEMICOLON(31),
							TokenNameQUESTION(24),
							TokenNameCOLON(81),
							TokenNameCOMMA(32),
							TokenNameDOT(3),
							TokenNameEQUAL(67),
							TokenNameAT(37),
							TokenNameELLIPSIS(127),
							TokenNameARROW(125),
							TokenNameCOLON_COLON(10),
							TokenNameBeginLambda(85),
							TokenNameBeginIntersectionCast(66),
							TokenNameBeginTypeArguments(98),
							TokenNameAT308(30),
							TokenNameAT308DOTDOTDOT(134),
							TokenNameCaseArrow(89),
							TokenNameRestrictedIdentifierYield(99),
							TokenNameRestrictedIdentifierrecord(84),
							TokenNameRestrictedIdentifiersealed(50),
							TokenNameRestrictedIdentifierpermits(128),
							TokenNameBeginCasePattern(129),
							TokenNameRestrictedIdentifierWhen(130),
							TokenNameUNDERSCORE(65),
							TokenNameEOF(68),
							TokenNameERROR(137);

// END_AUTOGENERATED_REGION

	private final static Map<Integer, TerminalToken> tokenMap = new HashMap<>(256);
	static {
		for (TerminalToken t : TerminalToken.values()) {
			tokenMap.put(t.tokenNumber(), t);
		}
	}

	private final int tokenNumber;

	// Constructor
	TerminalToken(int number) {
		this.tokenNumber = number;
	}

	public int tokenNumber() {
		return this.tokenNumber;
	}

	public static boolean isRestrictedKeyword(TerminalToken tokenType) {
		return switch (tokenType) {
			case TokenNameRestrictedIdentifierYield, TokenNameRestrictedIdentifierrecord, TokenNameRestrictedIdentifierWhen,
					TokenNameRestrictedIdentifiersealed, TokenNameRestrictedIdentifierpermits -> true;
			default -> false;
		};
	}

	public static TerminalToken getRestrictedKeyword(char [] text) {
		if (text != null) {
			int len = text.length;
			if (len == 4 && text[0] == 'w' ||
				len == 5 && text[0] == 'y' ||
				len == 6 && (text[0] == 'r' || text[0] == 's') ||
				len == 7 && text[0] == 'p') {
				return getRestrictedKeyword(new String(text));
			}
		}
		return TokenNameNotAToken;
	}

	public static TerminalToken getRestrictedKeyword(String text) {
		return switch (text) {
			case "yield"   -> TokenNameRestrictedIdentifierYield;   //$NON-NLS-1$
			case "record"  -> TokenNameRestrictedIdentifierrecord;  //$NON-NLS-1$
			case "when"    -> TokenNameRestrictedIdentifierWhen;    //$NON-NLS-1$
			case "sealed"  -> TokenNameRestrictedIdentifiersealed;  //$NON-NLS-1$
			case "permits" -> TokenNameRestrictedIdentifierpermits; //$NON-NLS-1$
			default        -> TokenNameNotAToken;
		};
	}

	public static TerminalToken of(int act) {
		TerminalToken token = tokenMap.get(act);
		if (token == null) {
			// Really shouldn't occur -- perhaps if parser non-terminals are looked up?
			throw new IllegalArgumentException("Unknown token number = " + act); //$NON-NLS-1$
		}
		return token;
	}

	static TerminalToken maybeOf(int act) {
		return tokenMap.get(act);
	}

}

