/**
 * Copyright (c) 2017, 2020 Kichwa Coders Ltd. and others.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */
package org.eclipse.lsp4j.debug;

import org.eclipse.lsp4j.jsonrpc.ProtocolSince;
import org.eclipse.lsp4j.jsonrpc.util.Preconditions;
import org.eclipse.lsp4j.jsonrpc.util.ToStringBuilder;
import org.eclipse.lsp4j.jsonrpc.validation.NonNull;

/**
 * Response to 'evaluate' request.
 */
@SuppressWarnings("all")
public class EvaluateResponse {
  /**
   * The result of the evaluate request.
   */
  @NonNull
  private String result;

  /**
   * The type of the evaluate result.
   * <p>
   * This attribute should only be returned by a debug adapter if the corresponding capability
   * {@link InitializeRequestArguments#getSupportsVariableType} is true.
   * <p>
   * This is an optional property.
   */
  private String type;

  /**
   * Properties of an evaluate result that can be used to determine how to render the result in the UI.
   * <p>
   * This is an optional property.
   */
  private VariablePresentationHint presentationHint;

  /**
   * If `variablesReference` is &gt; 0, the evaluate result is structured and its children can be retrieved by passing
   * `variablesReference` to the `variables` request as long as execution remains suspended.
   * See 'Lifetime of Object References' in the {@link DebugProtocol#Overview} section for details.
   * <p>
   * The value should be less than or equal to 2147483647 (2^31-1).
   */
  private int variablesReference;

  /**
   * The number of named child variables.
   * <p>
   * The client can use this information to present the variables in a paged UI and fetch them in chunks.
   * <p>
   * The value should be less than or equal to 2147483647 (2^31-1).
   * <p>
   * This is an optional property.
   */
  private Integer namedVariables;

  /**
   * The number of indexed child variables.
   * <p>
   * The client can use this information to present the variables in a paged UI and fetch them in chunks.
   * <p>
   * The value should be less than or equal to 2147483647 (2^31-1).
   * <p>
   * This is an optional property.
   */
  private Integer indexedVariables;

  /**
   * A memory reference to a location appropriate for this result.
   * <p>
   * For pointer type eval results, this is generally a reference to the memory address contained in the pointer.
   * <p>
   * This attribute may be returned by a debug adapter if corresponding capability
   * {@link InitializeRequestArguments#getSupportsMemoryReferences} is true.
   * <p>
   * This is an optional property.
   */
  private String memoryReference;

  /**
   * A reference that allows the client to request the location where the returned value is declared. For example,
   * if a function pointer is returned, the adapter may be able to look up the function's location.
   * This should be present only if the adapter is likely to be able to resolve the location.
   * <p>
   * This reference shares the same lifetime as the `variablesReference`.
   * See 'Lifetime of Object References' in the Overview section for details.
   * <p>
   * This is an optional property.
   */
  @ProtocolSince("1.68")
  private Integer valueLocationReference;

  /**
   * The result of the evaluate request.
   */
  @NonNull
  public String getResult() {
    return this.result;
  }

  /**
   * The result of the evaluate request.
   */
  public void setResult(@NonNull final String result) {
    this.result = Preconditions.checkNotNull(result, "result");
  }

  /**
   * The type of the evaluate result.
   * <p>
   * This attribute should only be returned by a debug adapter if the corresponding capability
   * {@link InitializeRequestArguments#getSupportsVariableType} is true.
   * <p>
   * This is an optional property.
   */
  public String getType() {
    return this.type;
  }

  /**
   * The type of the evaluate result.
   * <p>
   * This attribute should only be returned by a debug adapter if the corresponding capability
   * {@link InitializeRequestArguments#getSupportsVariableType} is true.
   * <p>
   * This is an optional property.
   */
  public void setType(final String type) {
    this.type = type;
  }

  /**
   * Properties of an evaluate result that can be used to determine how to render the result in the UI.
   * <p>
   * This is an optional property.
   */
  public VariablePresentationHint getPresentationHint() {
    return this.presentationHint;
  }

  /**
   * Properties of an evaluate result that can be used to determine how to render the result in the UI.
   * <p>
   * This is an optional property.
   */
  public void setPresentationHint(final VariablePresentationHint presentationHint) {
    this.presentationHint = presentationHint;
  }

  /**
   * If `variablesReference` is &gt; 0, the evaluate result is structured and its children can be retrieved by passing
   * `variablesReference` to the `variables` request as long as execution remains suspended.
   * See 'Lifetime of Object References' in the {@link DebugProtocol#Overview} section for details.
   * <p>
   * The value should be less than or equal to 2147483647 (2^31-1).
   */
  public int getVariablesReference() {
    return this.variablesReference;
  }

  /**
   * If `variablesReference` is &gt; 0, the evaluate result is structured and its children can be retrieved by passing
   * `variablesReference` to the `variables` request as long as execution remains suspended.
   * See 'Lifetime of Object References' in the {@link DebugProtocol#Overview} section for details.
   * <p>
   * The value should be less than or equal to 2147483647 (2^31-1).
   */
  public void setVariablesReference(final int variablesReference) {
    this.variablesReference = variablesReference;
  }

  /**
   * The number of named child variables.
   * <p>
   * The client can use this information to present the variables in a paged UI and fetch them in chunks.
   * <p>
   * The value should be less than or equal to 2147483647 (2^31-1).
   * <p>
   * This is an optional property.
   */
  public Integer getNamedVariables() {
    return this.namedVariables;
  }

  /**
   * The number of named child variables.
   * <p>
   * The client can use this information to present the variables in a paged UI and fetch them in chunks.
   * <p>
   * The value should be less than or equal to 2147483647 (2^31-1).
   * <p>
   * This is an optional property.
   */
  public void setNamedVariables(final Integer namedVariables) {
    this.namedVariables = namedVariables;
  }

  /**
   * The number of indexed child variables.
   * <p>
   * The client can use this information to present the variables in a paged UI and fetch them in chunks.
   * <p>
   * The value should be less than or equal to 2147483647 (2^31-1).
   * <p>
   * This is an optional property.
   */
  public Integer getIndexedVariables() {
    return this.indexedVariables;
  }

  /**
   * The number of indexed child variables.
   * <p>
   * The client can use this information to present the variables in a paged UI and fetch them in chunks.
   * <p>
   * The value should be less than or equal to 2147483647 (2^31-1).
   * <p>
   * This is an optional property.
   */
  public void setIndexedVariables(final Integer indexedVariables) {
    this.indexedVariables = indexedVariables;
  }

  /**
   * A memory reference to a location appropriate for this result.
   * <p>
   * For pointer type eval results, this is generally a reference to the memory address contained in the pointer.
   * <p>
   * This attribute may be returned by a debug adapter if corresponding capability
   * {@link InitializeRequestArguments#getSupportsMemoryReferences} is true.
   * <p>
   * This is an optional property.
   */
  public String getMemoryReference() {
    return this.memoryReference;
  }

  /**
   * A memory reference to a location appropriate for this result.
   * <p>
   * For pointer type eval results, this is generally a reference to the memory address contained in the pointer.
   * <p>
   * This attribute may be returned by a debug adapter if corresponding capability
   * {@link InitializeRequestArguments#getSupportsMemoryReferences} is true.
   * <p>
   * This is an optional property.
   */
  public void setMemoryReference(final String memoryReference) {
    this.memoryReference = memoryReference;
  }

  /**
   * A reference that allows the client to request the location where the returned value is declared. For example,
   * if a function pointer is returned, the adapter may be able to look up the function's location.
   * This should be present only if the adapter is likely to be able to resolve the location.
   * <p>
   * This reference shares the same lifetime as the `variablesReference`.
   * See 'Lifetime of Object References' in the Overview section for details.
   * <p>
   * This is an optional property.
   */
  @ProtocolSince("1.68")
  public Integer getValueLocationReference() {
    return this.valueLocationReference;
  }

  /**
   * A reference that allows the client to request the location where the returned value is declared. For example,
   * if a function pointer is returned, the adapter may be able to look up the function's location.
   * This should be present only if the adapter is likely to be able to resolve the location.
   * <p>
   * This reference shares the same lifetime as the `variablesReference`.
   * See 'Lifetime of Object References' in the Overview section for details.
   * <p>
   * This is an optional property.
   */
  @ProtocolSince("1.68")
  public void setValueLocationReference(final Integer valueLocationReference) {
    this.valueLocationReference = valueLocationReference;
  }

  @Override
  public String toString() {
    ToStringBuilder b = new ToStringBuilder(this);
    b.add("result", this.result);
    b.add("type", this.type);
    b.add("presentationHint", this.presentationHint);
    b.add("variablesReference", this.variablesReference);
    b.add("namedVariables", this.namedVariables);
    b.add("indexedVariables", this.indexedVariables);
    b.add("memoryReference", this.memoryReference);
    b.add("valueLocationReference", this.valueLocationReference);
    return b.toString();
  }

  @Override
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    EvaluateResponse other = (EvaluateResponse) obj;
    if (this.result == null) {
      if (other.result != null)
        return false;
    } else if (!this.result.equals(other.result))
      return false;
    if (this.type == null) {
      if (other.type != null)
        return false;
    } else if (!this.type.equals(other.type))
      return false;
    if (this.presentationHint == null) {
      if (other.presentationHint != null)
        return false;
    } else if (!this.presentationHint.equals(other.presentationHint))
      return false;
    if (other.variablesReference != this.variablesReference)
      return false;
    if (this.namedVariables == null) {
      if (other.namedVariables != null)
        return false;
    } else if (!this.namedVariables.equals(other.namedVariables))
      return false;
    if (this.indexedVariables == null) {
      if (other.indexedVariables != null)
        return false;
    } else if (!this.indexedVariables.equals(other.indexedVariables))
      return false;
    if (this.memoryReference == null) {
      if (other.memoryReference != null)
        return false;
    } else if (!this.memoryReference.equals(other.memoryReference))
      return false;
    if (this.valueLocationReference == null) {
      if (other.valueLocationReference != null)
        return false;
    } else if (!this.valueLocationReference.equals(other.valueLocationReference))
      return false;
    return true;
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((this.result== null) ? 0 : this.result.hashCode());
    result = prime * result + ((this.type== null) ? 0 : this.type.hashCode());
    result = prime * result + ((this.presentationHint== null) ? 0 : this.presentationHint.hashCode());
    result = prime * result + this.variablesReference;
    result = prime * result + ((this.namedVariables== null) ? 0 : this.namedVariables.hashCode());
    result = prime * result + ((this.indexedVariables== null) ? 0 : this.indexedVariables.hashCode());
    result = prime * result + ((this.memoryReference== null) ? 0 : this.memoryReference.hashCode());
    return prime * result + ((this.valueLocationReference== null) ? 0 : this.valueLocationReference.hashCode());
  }
}
