/**
 * Copyright (c) 2016-2018 TypeFox and others.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */
package org.eclipse.lsp4j;

import org.eclipse.lsp4j.jsonrpc.ProtocolDraft;
import org.eclipse.lsp4j.jsonrpc.ProtocolSince;
import org.eclipse.lsp4j.jsonrpc.util.Preconditions;
import org.eclipse.lsp4j.jsonrpc.util.ToStringBuilder;
import org.eclipse.lsp4j.jsonrpc.validation.NonNull;

/**
 * Documentation for a class of code actions.
 */
@ProtocolDraft
@ProtocolSince("3.18.0")
@SuppressWarnings("all")
public class CodeActionKindDocumentation {
  /**
   * The kind of the code action being documented.
   * <p>
   * If the kind is generic, such as {@link CodeActionKind#Refactor}, the
   * documentation will be shown whenever any refactorings are returned. If
   * the kind is more specific, such as {@link CodeActionKind#RefactorExtract},
   * the documentation will only be shown when extract refactoring code actions
   * are returned.
   */
  @NonNull
  private String kind;

  /**
   * Command that is used to display the documentation to the user.
   * <p>
   * The title of this documentation code action is taken
   * from {@link Command#title}.
   */
  @NonNull
  private Command command;

  /**
   * The kind of the code action being documented.
   * <p>
   * If the kind is generic, such as {@link CodeActionKind#Refactor}, the
   * documentation will be shown whenever any refactorings are returned. If
   * the kind is more specific, such as {@link CodeActionKind#RefactorExtract},
   * the documentation will only be shown when extract refactoring code actions
   * are returned.
   */
  @NonNull
  public String getKind() {
    return this.kind;
  }

  /**
   * The kind of the code action being documented.
   * <p>
   * If the kind is generic, such as {@link CodeActionKind#Refactor}, the
   * documentation will be shown whenever any refactorings are returned. If
   * the kind is more specific, such as {@link CodeActionKind#RefactorExtract},
   * the documentation will only be shown when extract refactoring code actions
   * are returned.
   */
  public void setKind(@NonNull final String kind) {
    this.kind = Preconditions.checkNotNull(kind, "kind");
  }

  /**
   * Command that is used to display the documentation to the user.
   * <p>
   * The title of this documentation code action is taken
   * from {@link Command#title}.
   */
  @NonNull
  public Command getCommand() {
    return this.command;
  }

  /**
   * Command that is used to display the documentation to the user.
   * <p>
   * The title of this documentation code action is taken
   * from {@link Command#title}.
   */
  public void setCommand(@NonNull final Command command) {
    this.command = Preconditions.checkNotNull(command, "command");
  }

  @Override
  public String toString() {
    ToStringBuilder b = new ToStringBuilder(this);
    b.add("kind", this.kind);
    b.add("command", this.command);
    return b.toString();
  }

  @Override
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    CodeActionKindDocumentation other = (CodeActionKindDocumentation) obj;
    if (this.kind == null) {
      if (other.kind != null)
        return false;
    } else if (!this.kind.equals(other.kind))
      return false;
    if (this.command == null) {
      if (other.command != null)
        return false;
    } else if (!this.command.equals(other.command))
      return false;
    return true;
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((this.kind== null) ? 0 : this.kind.hashCode());
    return prime * result + ((this.command== null) ? 0 : this.command.hashCode());
  }
}
