"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProtocolClient = void 0;
const ee = require("events");
class ProtocolClient extends ee.EventEmitter {
    constructor() {
        super();
        this.pendingRequests = new Map();
        this.rawData = Buffer.alloc(0);
        this.sequence = 1;
        this.contentLength = -1;
    }
    connect(readable, writable) {
        this.outputStream = writable;
        readable.on('data', (data) => {
            this.handleData(data);
        });
    }
    send(command, args) {
        return new Promise((completeDispatch, errorDispatch) => {
            this.doSend(command, args, (result) => {
                if (result.success) {
                    completeDispatch(result);
                }
                else {
                    errorDispatch(new Error(result.message));
                }
            });
        });
    }
    doSend(command, args, clb) {
        const request = {
            type: 'request',
            seq: this.sequence++,
            command: command
        };
        if (args && Object.keys(args).length > 0) {
            request.arguments = args;
        }
        // store callback for this request
        this.pendingRequests.set(request.seq, clb);
        const json = JSON.stringify(request);
        this.outputStream.write(`Content-Length: ${Buffer.byteLength(json, 'utf8')}\r\n\r\n${json}`, 'utf8');
    }
    handleData(data) {
        this.rawData = Buffer.concat([this.rawData, data]);
        while (true) {
            if (this.contentLength >= 0) {
                if (this.rawData.length >= this.contentLength) {
                    const message = this.rawData.toString('utf8', 0, this.contentLength);
                    this.rawData = this.rawData.slice(this.contentLength);
                    this.contentLength = -1;
                    if (message.length > 0) {
                        this.dispatch(message);
                    }
                    continue; // there may be more complete messages to process
                }
            }
            else {
                const idx = this.rawData.indexOf(ProtocolClient.TWO_CRLF);
                if (idx !== -1) {
                    const header = this.rawData.toString('utf8', 0, idx);
                    const lines = header.split('\r\n');
                    for (let i = 0; i < lines.length; i++) {
                        const pair = lines[i].split(/: +/);
                        if (pair[0] === 'Content-Length') {
                            this.contentLength = +pair[1];
                        }
                    }
                    this.rawData = this.rawData.slice(idx + ProtocolClient.TWO_CRLF.length);
                    continue;
                }
            }
            break;
        }
    }
    dispatch(body) {
        const rawData = JSON.parse(body);
        if (typeof rawData.event !== 'undefined') {
            const event = rawData;
            this.emit(event.event, event);
        }
        else {
            const response = rawData;
            const clb = this.pendingRequests.get(response.request_seq);
            if (clb) {
                this.pendingRequests.delete(response.request_seq);
                clb(response);
            }
        }
    }
}
exports.ProtocolClient = ProtocolClient;
ProtocolClient.TWO_CRLF = '\r\n\r\n';
//# sourceMappingURL=data:application/json;base64,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